package com.aminocom.playerdemo;

import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.util.Pair;
import android.view.KeyEvent;
import android.view.SurfaceView;
import android.view.View;
import android.widget.TextView;

import androidx.appcompat.app.AppCompatActivity;
import androidx.lifecycle.Observer;

import com.aminocom.libplayer.PlaybackInfo;
import com.aminocom.libplayer.PlayerEvent;
import com.aminocom.libplayer.SandboxedPlayer;
import com.aminocom.libplayer.SystemUtil;
import com.aminocom.libplayer.TrackInfo;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class MainActivity extends AppCompatActivity {
    private static final String TAG = "demo";
    private static final int HINT_DURATION = 5000; // 5 seconds
    private final HashMap<Integer, SandboxedPlayer> playerMap = new HashMap<>();
    private SandboxedPlayer mPlayer = null;
    private SystemUtil systemUtil = null;
    private TextView positionText, volumeText, speedText, hintOverlay, channelName, channelDesc;
    private int resizeMode = 0;
    private int displayCorner = 0;
    private int rotate = 0;

    private final List<Channel> channels = new ArrayList<>();
    Observer<Boolean> mObserver = isConnected -> {
        if (isConnected) {
            parseChannels();
        }
    };
    private int currentChannelIndex = 0;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        setupConnection();

        // Bind views
        positionText = findViewById(R.id.position_text);
        volumeText = findViewById(R.id.volume_text);
        speedText = findViewById(R.id.speed_text);
        hintOverlay = findViewById(R.id.hint_overlay);
        channelName = findViewById(R.id.channel_name);
        channelDesc = findViewById(R.id.channel_desc);
        showHint();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        playerMap.values().forEach(SandboxedPlayer::release);
        playerMap.clear();
    }

    private void setupConnection() {
        systemUtil = new SystemUtil(this);
        systemUtil.setObserver(this, mObserver);
        initPlayer(R.id.surfaceView1, "dolbyio://k9Mwad/multiview");
        //initPlayer(R.id.surfaceView2, "udp://233.22.133.12:8110");
        //initPlayer(R.id.surfaceView3, "http://demo-dash-vod.zahs.tv/exodus/manifest.mpd");
        //initPlayer(R.id.surfaceView4, "dolbyio://k9Mwad/LiveDealerDemo");
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        switch (keyCode) {
            case KeyEvent.KEYCODE_DPAD_RIGHT:
                mPlayer.seekTo(mPlayer.getPlaybackInfo().position + 10000);
                updateUI();
                return true;
            case KeyEvent.KEYCODE_DPAD_LEFT:
                mPlayer.seekTo(Math.max(0, mPlayer.getPlaybackInfo().position - 10000));
                updateUI();
                return true;
            case KeyEvent.KEYCODE_DPAD_UP:
                mPlayer.setVolume(Math.min(1.0f, mPlayer.getVolume() + 0.1f));
                updateUI();
                return true;
            case KeyEvent.KEYCODE_DPAD_DOWN:
                mPlayer.setVolume(Math.max(0.0f, mPlayer.getVolume() - 0.1f));
                updateUI();
                return true;
            case KeyEvent.KEYCODE_INFO:
                toggleHint();
                return true;
            case KeyEvent.KEYCODE_CHANNEL_UP:
                loadChannel(currentChannelIndex + 1);
                return true;
            case KeyEvent.KEYCODE_CHANNEL_DOWN:
                loadChannel(currentChannelIndex - 1);
                return true;
            case KeyEvent.KEYCODE_MEDIA_PLAY:
                mPlayer.setPlaybackSpeed(1.0f);
                updateUI();
                return true;
            case KeyEvent.KEYCODE_MEDIA_PAUSE:
                mPlayer.setPlaybackSpeed(0.0f);
                updateUI();
                return true;
            case KeyEvent.KEYCODE_DPAD_CENTER:
                // Toggle Video Track
                TrackInfo info = mPlayer.getVideoTrackInfo();
                int id = info.getSelectedIndex();
                if (info.getTracks().size() > 1)
                    mPlayer.setVideoTrack((id+1)%info.getTracks().size());
                updateUI();
                return true;
            case KeyEvent.KEYCODE_MEDIA_STOP:
                mPlayer.reset();
                updateUI();
                return true;
            case KeyEvent.KEYCODE_1:
                boolean blank = mPlayer.getBlank();
                mPlayer.setBlank(!blank);
                Log.d(TAG, "setBlank " + !blank);
                return true;
            case KeyEvent.KEYCODE_2:
                boolean isLive = mPlayer.isLiveStream();
                Log.d(TAG, "isLiveStream " + isLive);
                return true;
            case KeyEvent.KEYCODE_3:
                boolean audioEnable = mPlayer.getAudioRendering();
                mPlayer.setAudioRendering(!audioEnable);
                Log.d(TAG, "setAudioRendering " + !audioEnable);
                return true;
            case KeyEvent.KEYCODE_4:
                mPlayer.setResizeMode(resizeMode);
                Log.d(TAG, "setResizeMode " + resizeMode);
                resizeMode = (resizeMode + 1) % 5;
                return true;
            case KeyEvent.KEYCODE_5:
                displayCorner = (displayCorner + 1) % 5;

                // Each quadrant: scale to 2x and shift to one corner
                if (displayCorner == 1) {
                    mPlayer.scaleAndTranslate(2f, 2f, -960f, -540f); // Top-left
                } else if (displayCorner == 2) {
                    mPlayer.scaleAndTranslate(2f, 2f, 960f, -540f); // Top-right
                } else if (displayCorner == 3) {
                    mPlayer.scaleAndTranslate(2f, 2f, -960f, 540f); // Bottom-left
                } else if (displayCorner == 4) {
                    mPlayer.scaleAndTranslate(2f, 2f, 960f, 540f); // Bottom-right
                } else {
                    // Reset all transformations
                    mPlayer.scaleAndTranslate(1f, 1f, 0f, 0f);
                }
                return true;

            case KeyEvent.KEYCODE_6:
                rotate = (rotate + 1) % 12; // 0 to 330 in steps of 30
                float angle = rotate * 30f;

                mPlayer.rotate(angle); // Server will automatically reset scale/translate
                return true;
        }
        return super.onKeyDown(keyCode, event);
    }

    private void updateUI() {
        PlaybackInfo info = mPlayer.getPlaybackInfo();
        new Handler(Looper.getMainLooper()).postDelayed(this::updateUI, 500);
        runOnUiThread(() -> {
            positionText.setText(String.format("Position: %d\nDuration: %d\nBuffered Duration: %d", info.position, info.duration, info.bufferedDuration));
            volumeText.setText(String.format("Volume: %.1f", mPlayer.getVolume()));
            speedText.setText(String.format("Speed: %.1f", mPlayer.getPlaybackSpeed()));
        });
    }

    private void showHint() {
        runOnUiThread(() -> hintOverlay.setVisibility(View.VISIBLE));
    }

    private void toggleHint() {
        if (hintOverlay.getVisibility() == View.VISIBLE) {
            hintOverlay.setVisibility(View.GONE);
        } else {
            showHint();
        }
    }

    private void initPlayer(int surfaceViewId, String url) {
        Context context = this;
        SurfaceView surfaceView = findViewById(surfaceViewId);
        if (surfaceView == null) {
            Log.e(TAG, "SurfaceView not found for ID: " + surfaceViewId);
            return;
        }
        SandboxedPlayer player;
        if (!playerMap.containsKey(surfaceViewId)) {
            player = new SandboxedPlayer(context, surfaceView);
            playerMap.put(surfaceViewId, player);
        } else {
            player = playerMap.get(surfaceViewId);
        }

        assert player != null;
        if (mPlayer == null) {
            mPlayer = player;
        }
        player.addEventListener((event, bundle) -> {
            Log.d("Client", "Received event: " + event + " for player " + player.getPlayerId());
            switch (event) {
                case PlayerEvent.PLAYBACK_STARTED:
                    Pair<String, Bundle> result = player.getVideoSource();
                    Log.e("app", "current source:" + result.first);
                    //updateUI();
                    break;
            }
        });
        player.loadVideoSource(url, null);
    }

    private void parseChannels() {
        String jsonString = systemUtil.getIni("webapp.lineup_json", "");
        if (jsonString.isEmpty()) {
            Log.e(TAG, "webapp.lineup_json is Empty");
            return;
        }

        channels.clear();
        try {
            JSONArray jsonArray = new JSONArray(jsonString);
            for (int i = 0; i < jsonArray.length(); i++) {
                JSONObject obj = jsonArray.getJSONObject(i);
                String name = obj.getString("name");
                String desc = obj.getString("desc");
                String url;

                if (obj.get("url") instanceof JSONArray) {
                    url = obj.getJSONArray("url").getString(0); // Take first URL if array
                } else {
                    url = obj.getString("url");
                }

                channels.add(new Channel(name, desc, url));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void loadChannel(int index) {
        if (channels.isEmpty() || mPlayer == null) {
            return;
        }

        if (index >= channels.size()) {
            index = 0;
        } else if (index < 0) {
            index = channels.size() - 1;
        }
        currentChannelIndex = index;
        Channel channel = channels.get(index);

        mPlayer.loadVideoSource(channel.url, null);
        runOnUiThread(() -> {
            channelName.setText(channel.name);
            channelDesc.setText(channel.desc);
        });
    }

    // Channel class to hold parsed data
    private static class Channel {
        String name;
        String desc;
        String url; // Single URL or first URL if array

        Channel(String name, String desc, String url) {
            this.name = name;
            this.desc = desc;
            this.url = url;
        }
    }
};